-- ============================================================================
-- AutoOpenCore.lua - Core Logic for AutoOpen System
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Intelligent player/vehicle detection
-- - Configurable user attributes per trigger
-- - Farm-based security checks
-- - Simple counter system for collision detection
-- - Multiplayer synchronization events
-- - Automatic timer system
-- - Complete GUI settings integration
-- - Robust boolean user attribute handling with error catching
-- - Trigger name logging support
-- ============================================================================

AutoOpen = {}

-- ============================================================================
-- Helper Functions
-- ============================================================================

-- Helper function to get AutoOpen Manager
local function getAutoOpenManager()
    if g_currentMission and g_currentMission.autoOpen then
        return g_currentMission.autoOpen
    end
    debugPrint("WARNING: AutoOpen Manager not available!")
    return nil
end

-- Helper function to get trigger name for logging
local function getTriggerName(triggerId)
    if triggerId == nil or triggerId == 0 then
        return "[Unknown]"
    end
    
    local success, name = pcall(function()
        return getName(triggerId)
    end)
    
    if success and name and name ~= "" then
        return "[" .. name .. "]"
    else
        return "[Trigger_" .. tostring(triggerId) .. "]"
    end
end

-- Robust boolean conversion for user attributes with fallback
local function getUserAttributeBoolean(nodeId, attributeName, defaultValue)
    if defaultValue == nil then 
        defaultValue = true 
    end
    
    local attr = getUserAttribute(nodeId, attributeName)
    
    -- Real boolean values (correct usage)
    if type(attr) == "boolean" then
        debugPrint("UserAttribute " .. attributeName .. " (boolean): " .. tostring(attr))
        return attr
    end
    
    -- String fallback for input errors
    if type(attr) == "string" then
        local lower = string.lower(attr)
        debugPrint("UserAttribute " .. attributeName .. " (string fallback): '" .. attr .. "'")
        
        -- TRUE variants
        if lower == "true" or lower == "1" or lower == "yes" or lower == "ja" then
            debugPrint("String fallback: interpreted as TRUE")
            return true
        end
        
        -- FALSE variants  
        if lower == "false" or lower == "0" or lower == "no" or lower == "nein" then
            debugPrint("String fallback: interpreted as FALSE")
            return false
        end
        
        -- Unknown string values → Default = true
        debugPrint("WARNING: Unknown string value for " .. attributeName .. " - using default: true")
        return true
    end
    
    -- nil or other types → Default = true
    if attr == nil then
        debugPrint("UserAttribute " .. attributeName .. " not set - using default: true")
    else
        debugPrint("WARNING: Invalid type for " .. attributeName .. " - using default: true")
    end
    
    return true
end

-- ============================================================================
-- Main AutoOpen Functions
-- ============================================================================

function AutoOpen:autoOpenSet(animatedObject, open, noEventSend)
    animatedObject.lastGateState = animatedObject.lastGateState or false
    animatedObject.lastEventTime = animatedObject.lastEventTime or 0
    local currentTime = g_currentMission.time
    
    local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
    debugPrint("autoOpenSet called for trigger: " .. triggerName)
    
    -- Use intelligent gate control everywhere
    local isMultiplayer = g_server ~= nil or g_client ~= nil
    
    if isMultiplayer and not noEventSend then
        -- MULTIPLAYER: Send event first, then wait
        local stateChanged = open ~= animatedObject.lastGateState
        local timeToRefresh = (currentTime - animatedObject.lastEventTime) > 1000
        
        if stateChanged or timeToRefresh then
            animatedObject.lastGateState = open
            animatedObject.lastEventTime = currentTime
            
            debugPrint("Sending event for trigger: " .. triggerName)
            
            local eventType = open and AutoOpenEvent.TYPE_OPEN or AutoOpenEvent.TYPE_CLOSE
            AutoOpenEvent.sendEvent(animatedObject, eventType)
            
            -- Only server starts immediately, clients wait for event
            if g_server ~= nil then
                -- SERVER: Set desired direction explicitly
                debugPrint("SERVER: Setting animation explicitly for trigger: " .. triggerName)
                
                local direction = open and 1 or -1
                debugPrint(string.format("SERVER: Gate %s is being %s", triggerName, open and "opened" or "closed"))
                
                -- IMPORTANT: Direction reversal check for sound restart
                if animatedObject.lastDirection and animatedObject.lastDirection ~= direction then
                    debugPrint(string.format("SERVER: Direction reversal for %s - from %s to %s!", 
                        triggerName,
                        animatedObject.lastDirection == 1 and "opening" or "closing",
                        direction == 1 and "opening" or "closing"))
                    debugPrint("SERVER: Starting new sound cycle (direction change or start) for " .. triggerName)
                    -- Stop current sound completely
                    if animatedObject.sharedLoadRequestIds ~= nil then
                        for _, sampleElement in pairs(animatedObject.sharedLoadRequestIds) do
                            if sampleElement.sample ~= nil then
                                g_soundManager:stopSample(sampleElement.sample)
                            end
                        end
                    end
                else
                    debugPrint("SERVER: First activation or same direction for " .. triggerName)
                    if not animatedObject.lastDirection then
                        debugPrint("SERVER: Starting new sound cycle (direction change or start) for " .. triggerName)
                    end
                end
                
                -- Save current direction for next comparison
                animatedObject.lastDirection = direction
                
                -- Detect direction change during animation
                local wasMoving = animatedObject.animation.direction ~= 0
                local directionChange = false
                
                if open then
                    if animatedObject.animation.time < 1 then
                        -- Check if we're changing direction
                        if animatedObject.animation.direction == -1 then
                            directionChange = true
                        end
                        animatedObject.animation.direction = 1
                    end
                else
                    if animatedObject.animation.time > 0 then
                        -- Check if we're changing direction
                        if animatedObject.animation.direction == 1 then
                            directionChange = true
                        end
                        animatedObject.animation.direction = -1
                    end
                end
                
                -- ALWAYS raiseActive() on direction change or start
                if not wasMoving or directionChange then
                    animatedObject:raiseActive()
                    
                    -- ADDITIONALLY: Force animation restart on direction reversal
                    if directionChange and animatedObject.playAnimation and animatedObject.animation and animatedObject.animation.name then
                        animatedObject:playAnimation(animatedObject.animation.name)
                    end
                end
            end
            -- CLIENTS start NOTHING - wait for event
        else
            debugPrint("Event not sent for trigger " .. triggerName .. " - state not changed or too early")
        end
    else
        -- SINGLE PLAYER or event received (noEventSend=true)
        debugPrint("SP mode or event received - using intelligent control for trigger: " .. triggerName)
        
        local direction = open and 1 or -1
        debugPrint(string.format("SP: Gate %s is being %s", triggerName, open and "opened" or "closed"))
        
        -- IMPORTANT: Direction reversal check for sound restart
        if animatedObject.lastDirection and animatedObject.lastDirection ~= direction then
            debugPrint(string.format("SP: Direction reversal for %s - from %s to %s!", 
                triggerName,
                animatedObject.lastDirection == 1 and "opening" or "closing",
                direction == 1 and "opening" or "closing"))
            debugPrint("SP: Starting new sound cycle (direction change or start) for " .. triggerName)
            -- Stop current sound completely
            if animatedObject.sharedLoadRequestIds ~= nil then
                for _, sampleElement in pairs(animatedObject.sharedLoadRequestIds) do
                    if sampleElement.sample ~= nil then
                        g_soundManager:stopSample(sampleElement.sample)
                    end
                end
            end
        else
            debugPrint("SP: First activation or same direction for " .. triggerName)
            if not animatedObject.lastDirection then
                debugPrint("SP: Starting new sound cycle (direction change or start) for " .. triggerName)
            end
        end
        
        -- Save current direction for next comparison
        animatedObject.lastDirection = direction
        
        -- Detect direction change during animation
        local wasMoving = animatedObject.animation.direction ~= 0
        local directionChange = false
        
        if open then
            if animatedObject.animation.time < 1 then
                -- Check if we're changing direction
                if animatedObject.animation.direction == -1 then
                    directionChange = true
                end
                animatedObject.animation.direction = 1
            end
        else
            if animatedObject.animation.time > 0 then
                -- Check if we're changing direction
                if animatedObject.animation.direction == 1 then
                    directionChange = true
                end
                animatedObject.animation.direction = -1
            end
        end
        
        -- ALWAYS raiseActive() on direction change or start
        if not wasMoving or directionChange then
            animatedObject:raiseActive()
            
            -- ADDITIONALLY: Force animation restart on direction reversal
            if directionChange and animatedObject.playAnimation and animatedObject.animation and animatedObject.animation.name then
                animatedObject:playAnimation(animatedObject.animation.name)
            end
        end
        
        animatedObject.lastGateState = open
        animatedObject.lastEventTime = currentTime
    end
end

function AutoOpen:initializeObjectCounter(animatedObject)
    if not animatedObject.autoOpenObjectCount then
        animatedObject.autoOpenObjectCount = 0
        local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
        debugPrint("ObjectCounter initialized for trigger: " .. triggerName)
    end
    if not animatedObject.trackedObjects then
        animatedObject.trackedObjects = {}
    end
    if not animatedObject.recentlyLeftObjects then
        animatedObject.recentlyLeftObjects = {}
    end
    if animatedObject.autoOpenObjectCount < 0 then
        animatedObject.autoOpenObjectCount = 0
        animatedObject.trackedObjects = {}
        animatedObject.recentlyLeftObjects = {}
        local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
        debugPrint("ObjectCounter reset (was negative) for trigger: " .. triggerName)
    end
end

function AutoOpen:addObjectToCounter(animatedObject, objectId, object)
    AutoOpen:initializeObjectCounter(animatedObject)
    
    local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
    debugPrint("Attempting to add object to trigger: " .. triggerName)
    
    local currentTime = g_currentMission.time
    local REENTER_COOLDOWN = 3000
    
    if animatedObject.recentlyLeftObjects[objectId] then
        local timeSinceLeft = currentTime - animatedObject.recentlyLeftObjects[objectId]
        if timeSinceLeft < REENTER_COOLDOWN then
            debugPrint("RE-ENTER PROTECTION - BLOCKED! (" .. timeSinceLeft .. "ms) for trigger: " .. triggerName)
            return false
        else
            animatedObject.recentlyLeftObjects[objectId] = nil
            debugPrint("RE-ENTER COOLDOWN expired (" .. timeSinceLeft .. "ms) for trigger: " .. triggerName)
        end
    end
    
    if animatedObject.trackedObjects[objectId] then
        debugPrint("Object already tracked - ignoring for trigger: " .. triggerName)
        return false
    end
    
    animatedObject.trackedObjects[objectId] = true
    animatedObject.autoOpenObjectCount = animatedObject.autoOpenObjectCount + 1
    
    debugPrint("Object added to trigger " .. triggerName .. " - New count: " .. tostring(animatedObject.autoOpenObjectCount))
    return true
end

function AutoOpen:removeObjectFromCounter(animatedObject, objectId, object)
    AutoOpen:initializeObjectCounter(animatedObject)
    
    local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
    debugPrint("Attempting to remove object from trigger: " .. triggerName)
    
    if not animatedObject.trackedObjects[objectId] then
        debugPrint("Object was not in tracker for trigger: " .. triggerName)
        return false
    end
    
    animatedObject.trackedObjects[objectId] = nil
    animatedObject.autoOpenObjectCount = animatedObject.autoOpenObjectCount - 1
    animatedObject.recentlyLeftObjects[objectId] = g_currentMission.time
    
    if animatedObject.autoOpenObjectCount < 0 then
        animatedObject.autoOpenObjectCount = 0
    end
    
    debugPrint("Object removed from trigger " .. triggerName .. " - New count: " .. tostring(animatedObject.autoOpenObjectCount))
    return true
end

function AutoOpen:shouldAutoOpen(animatedObject, object, triggerId)
    AutoOpen:initializeObjectCounter(animatedObject)
    
    local triggerName = getTriggerName(triggerId)
    debugPrint("shouldAutoOpen checked for trigger: " .. triggerName)
    
    -- GUI SETTINGS CHECK: MASTER SWITCH via Manager
    local manager = getAutoOpenManager()
    if not manager then
        debugPrint("AutoOpen Manager not available - using fallback for trigger: " .. triggerName)
        return false
    end
    
    -- Use correct setting names (old names for compatibility)
    if not manager:getSetting("autoOpenEnabled") then
        debugPrint("AutoOpen completely disabled (Master Switch) for trigger: " .. triggerName)
        return false
    end
    
    if object == nil or type(object) ~= "table" then
        debugPrint("Object is nil or not a table for trigger: " .. triggerName)
        return false
    end
    
    -- FARM ANALYSIS
    local buildingFarmId = animatedObject.ownerFarmId or animatedObject.farmId
    local myFarmId = g_currentMission.playerUserId and 
                     g_currentMission.playerSystem:getPlayerByUserId(g_currentMission.playerUserId).farmId or 1
    
    debugPrint("Farm check performed for trigger: " .. triggerName)
    
    if buildingFarmId ~= nil and buildingFarmId ~= 0 then
        if buildingFarmId ~= myFarmId then
            debugPrint(">>> ACCESS DENIED - Foreign building! <<< for trigger: " .. triggerName)
            return false
        else
            debugPrint("Farm authorization OK for trigger: " .. triggerName)
        end
    else
        debugPrint("Gate has no valid FarmId - allowing for trigger: " .. triggerName)
    end
    
    -- ROBUST USER ATTRIBUTE HANDLING with error catching
    debugPrint("User attribute evaluation for trigger: " .. triggerName)
    local xmlUserSetting = getUserAttributeBoolean(triggerId, "autoOpenUser", true)
    local xmlVehicleSetting = getUserAttributeBoolean(triggerId, "autoOpenVehicle", true)
    debugPrint("XML Settings for trigger " .. triggerName .. ": User=" .. tostring(xmlUserSetting) .. " Vehicle=" .. tostring(xmlVehicleSetting))
    
    -- Get GUI Settings LIVE from Manager
    local finalUserSetting = xmlUserSetting and manager:getSetting("autoOpenPlayersEnabled")
    local finalVehicleSetting = xmlVehicleSetting and manager:getSetting("autoOpenVehiclesEnabled")
    
    local triggerSettings = {
        autoOpenUser = finalUserSetting,
        autoOpenVehicle = finalVehicleSetting
    }
    
    debugPrint("Current trigger settings for " .. triggerName .. ": autoOpenUser=" .. tostring(finalUserSetting) .. " autoOpenVehicle=" .. tostring(finalVehicleSetting))
    
    -- Object type determination
    local isPlayer = false
    local isVehicle = false
    
    local function safeCheck(obj, property)
        if obj == nil or type(obj) ~= "table" then
            return false, nil
        end
        local success, result = pcall(function()
            return obj[property]
        end)
        return success and result, result
    end
    
    local hasPlayer, _ = safeCheck(object, "isPlayer")
    local hasPlayerRef, _ = safeCheck(object, "player")
    local hasUserId, _ = safeCheck(object, "userId")
    local hasFarmId, farmId = safeCheck(object, "farmId")
    local hasGetOwnerFarmId, _ = safeCheck(object, "getOwnerFarmId")
    
    isPlayer = hasPlayer or hasPlayerRef or hasUserId or (hasFarmId and not hasGetOwnerFarmId)
    
    if not isPlayer then
        isVehicle = hasGetOwnerFarmId
    end
    
    debugPrint("Object type detected for trigger: " .. triggerName)
    
    if isVehicle then
        if not triggerSettings.autoOpenVehicle then
            debugPrint("Vehicle auto-open disabled (XML or GUI settings) for trigger: " .. triggerName)
            return false
        end
        
        local vehicleFarmId = 0
        local success, value = safeCheck(object, "ownerFarmId")
        if success and value then
            vehicleFarmId = value
        end
        
        debugPrint("Vehicle farm check for trigger: " .. triggerName)
        
        if vehicleFarmId ~= 0 and vehicleFarmId ~= myFarmId then
            debugPrint("Vehicle farm check failed for trigger: " .. triggerName)
            return false
        end
        if buildingFarmId and buildingFarmId ~= 0 and vehicleFarmId ~= buildingFarmId and vehicleFarmId ~= 0 then
            debugPrint("Vehicle building farm check failed for trigger: " .. triggerName)
            return false
        end
        
        debugPrint("Vehicle authorized for trigger: " .. triggerName)
        return true
        
    elseif isPlayer then
        if not triggerSettings.autoOpenUser then
            debugPrint("Player auto-open disabled (XML or GUI settings) for trigger: " .. triggerName)
            return false
        end
        
        local playerFarmId = farmId or 0
        
        debugPrint("Player farm check for trigger: " .. triggerName)
        
        if playerFarmId ~= 0 and playerFarmId ~= myFarmId then
            debugPrint("Player farm check failed for trigger: " .. triggerName)
            return false
        end
        if buildingFarmId and buildingFarmId ~= 0 and playerFarmId ~= buildingFarmId and playerFarmId ~= 0 then
            debugPrint("Player building farm check failed for trigger: " .. triggerName)
            return false
        end
        
        debugPrint("Player authorized for trigger: " .. triggerName)
        return true
    end
    
    debugPrint("Neither player nor vehicle detected for trigger: " .. triggerName)
    return false
end

function AutoOpen:hasObjectsInTrigger(animatedObject)
    AutoOpen:initializeObjectCounter(animatedObject)
    if animatedObject.autoOpenObjectCount < 0 then
        animatedObject.autoOpenObjectCount = 0
        return false
    end
    local hasObjects = animatedObject.autoOpenObjectCount > 0
    local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
    debugPrint("hasObjectsInTrigger checked for trigger " .. triggerName .. " - Count: " .. tostring(animatedObject.autoOpenObjectCount))
    return hasObjects
end

function AutoOpen:triggerCallback(animatedObject, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
    local success, errorMsg = pcall(function()
        local triggerName = getTriggerName(triggerId)
        debugPrint("Trigger callback start for trigger: " .. triggerName)
        
        if animatedObject.oldTriggerCallback then
            local oldSuccess, oldError = pcall(function()
                animatedObject.oldTriggerCallback(animatedObject, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
            end)
            if not oldSuccess then
                debugPrint("ERROR in oldTriggerCallback for trigger " .. triggerName .. ": " .. tostring(oldError))
            end
        end
        
        local currentTime = g_currentMission.time
        if not animatedObject.lastTriggerTime then
            animatedObject.lastTriggerTime = {}
        end
        
        local lastTime = animatedObject.lastTriggerTime[otherId] or 0
        local timeDiff = currentTime - lastTime
        
        if timeDiff < 500 then
            debugPrint("Rate-limited - SKIPPED (" .. timeDiff .. "ms) for trigger: " .. triggerName)
            return
        end
        
        animatedObject.lastTriggerTime[otherId] = currentTime
        
        if onEnter then
            debugPrint("ON ENTER for trigger: " .. triggerName)
            
            local object = g_currentMission.nodeToObject[otherId]
            debugPrint("Object mapping exists for trigger: " .. triggerName)
            
            if object == nil then
                debugPrint("NO OBJECT MAPPING for trigger: " .. triggerName)
                return
            end
            
            if type(object) ~= "table" then
                debugPrint("Object is not a table for trigger: " .. triggerName)
                return
            end
            
            debugPrint("Object is valid - checking shouldAutoOpen for trigger: " .. triggerName)
            local shouldOpen = AutoOpen:shouldAutoOpen(animatedObject, object, triggerId)
            debugPrint("shouldAutoOpen result determined for trigger: " .. triggerName)
            
            if shouldOpen then
                local wasAdded = AutoOpen:addObjectToCounter(animatedObject, otherId, object)
                debugPrint("Object added to counter for trigger: " .. triggerName)
                
                if wasAdded then
                    debugPrint("Calling autoOpenSet for trigger: " .. triggerName)
                    AutoOpen:autoOpenSet(animatedObject, true)
                    
                    -- Get delay setting via Manager
                    local manager = getAutoOpenManager()
                    local delaySeconds = manager and manager:getSetting("closeDelaySeconds") or 3
                    local delayMs = delaySeconds * 1000
                    
                    animatedObject.gateCloseTimer = currentTime + delayMs
                    animatedObject:raiseActive()
                    debugPrint("GATE OPENED! Timer set (" .. delayMs .. "ms) + raiseActive() called for trigger: " .. triggerName)
                end
            else
                debugPrint("No authorization - object will NOT be tracked for trigger: " .. triggerName)
            end
            
        elseif onLeave then
            debugPrint("ON LEAVE for trigger: " .. triggerName)
            
            local object = g_currentMission.nodeToObject[otherId]
            debugPrint("Object mapping exists for trigger: " .. triggerName)
            
            if object ~= nil and type(object) == "table" then
                local wasRemoved = AutoOpen:removeObjectFromCounter(animatedObject, otherId, object)
                if wasRemoved then
                    if not AutoOpen:hasObjectsInTrigger(animatedObject) then
                        local currentTime = g_currentMission.time
                        
                        -- Get delay setting via Manager
                        local manager = getAutoOpenManager()
                        local delaySeconds = manager and manager:getSetting("closeDelaySeconds") or 3
                        local delayMs = delaySeconds * 1000
                        
                        animatedObject.gateCloseTimer = currentTime + delayMs
                        animatedObject:raiseActive()
                        debugPrint("Close timer set (" .. delayMs .. "ms) + raiseActive() called for trigger: " .. triggerName)
                    else
                        animatedObject.gateCloseTimer = nil
                        debugPrint("Gate stays open - other objects in trigger: " .. triggerName)
                    end
                end
            else
                debugPrint("Object is nil/invalid on onLeave for trigger: " .. triggerName)
            end
        end
        
        debugPrint("Trigger callback end for trigger: " .. triggerName)
    end)
    
    if not success then
        local triggerName = getTriggerName(triggerId)
        debugPrint("CRITICAL ERROR in triggerCallback for trigger " .. triggerName .. ": " .. tostring(errorMsg))
    end
end

function AutoOpen:update(animatedObject, dt)
    if not animatedObject.autoOpenLastUpdateLog then
        animatedObject.autoOpenLastUpdateLog = 0
    end
    
    local currentTime = g_currentMission.time
    if currentTime - animatedObject.autoOpenLastUpdateLog > 30000 then
        animatedObject.autoOpenLastUpdateLog = currentTime
        local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
        debugPrint("Update for AnimatedObject with trigger: " .. triggerName)
        
        if animatedObject.recentlyLeftObjects then
            local CLEANUP_TIME = 15000
            for objectId, leaveTime in pairs(animatedObject.recentlyLeftObjects) do
                if currentTime - leaveTime > CLEANUP_TIME then
                    animatedObject.recentlyLeftObjects[objectId] = nil
                    debugPrint("Cleanup: RecentlyLeft object removed for trigger: " .. triggerName)
                end
            end
        end
    end
    
    if animatedObject.gateCloseTimer ~= nil then
        animatedObject:raiseActive()
    end
    
    if animatedObject.gateCloseTimer ~= nil then
        local currentTime = g_currentMission.time
        if currentTime > animatedObject.gateCloseTimer then
            local triggerName = getTriggerName(animatedObject.triggerNode or animatedObject.triggerId)
            debugPrint("Close timer expired for trigger: " .. triggerName)
            if not AutoOpen:hasObjectsInTrigger(animatedObject) then
                debugPrint("Closing gate - no more objects in trigger: " .. triggerName)
                
                local setSuccess, setError = pcall(function()
                    AutoOpen:autoOpenSet(animatedObject, false)
                end)
                if not setSuccess then
                    debugPrint("ERROR in autoOpenSet for trigger " .. triggerName .. ": " .. tostring(setError))
                end
                
                animatedObject.gateCloseTimer = nil
            else
                debugPrint("Gate stays open - still objects in trigger: " .. triggerName)
                
                -- Get delay setting via Manager  
                local manager = getAutoOpenManager()
                local delaySeconds = manager and manager:getSetting("closeDelaySeconds") or 3
                local delayMs = math.max(2000, delaySeconds * 500)
                
                animatedObject.gateCloseTimer = currentTime + delayMs
                animatedObject:raiseActive()
            end
        end
    end
end

debugPrint("AutoOpenCore System loaded - Manager integration active (v1.1.0.0)")