-- ============================================================================
-- AutoOpenHooks.lua - Hook Installation and System Integration
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - AnimatedObject hook installation
-- - GUI settings integration with mod-conflict protection
-- - Settings save/load hooks
-- - System initialization and cleanup
-- - Error handling for all hooks
-- - Robust GUI injection with fallback mechanisms
-- ============================================================================

AutoOpenHooks = {}

function AutoOpenHooks:installHooks()
    debugPrint("AutoOpen Hooks - Installation start")

    -- Original AutoOpen Hooks
    if AnimatedObject.triggerCallback ~= nil then
        debugPrint("AnimatedObject.triggerCallback already exists - overriding...")
        AnimatedObject.oldTriggerCallback = AnimatedObject.triggerCallback
        function AnimatedObject:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, anotherID)
            -- Safety check: Is AutoOpen available?
            if AutoOpen == nil then
                debugPrint("WARNING: AutoOpen not available - using original callback")
                if self.oldTriggerCallback then
                    self.oldTriggerCallback(self, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
                end
                return
            end
            
            AutoOpen:triggerCallback(self, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
        end
        debugPrint("triggerCallback successfully overridden!")
    else
        debugPrint("No existing triggerCallback - creating new...")
        function AnimatedObject:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, anotherID)
            -- Safety check: Is AutoOpen available?
            if AutoOpen == nil then
                debugPrint("WARNING: AutoOpen not available - no action")
                return
            end
            
            AutoOpen:triggerCallback(self, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
        end
        debugPrint("New triggerCallback created!")
    end

    local originalUpdate = AnimatedObject.update
    debugPrint("Update hook being installed...")

    function AnimatedObject:update(dt)
        if originalUpdate and type(originalUpdate) == "function" then
            local success, errorMsg = pcall(function()
                originalUpdate(self, dt)
            end)
            if not success then
                debugPrint("ERROR in original Update: " .. tostring(errorMsg))
            end
        end
        
        -- Safety check for AutoOpen Update
        if self.gateCloseTimer ~= nil and AutoOpen ~= nil then
            local success, errorMsg = pcall(function()
                AutoOpen:update(self, dt)
            end)
            if not success then
                debugPrint("ERROR in AutoOpen Update: " .. tostring(errorMsg))
            end
        end
    end

    debugPrint("Update hook installed!")
    
    -- GUI Hooks Installation with enhanced robustness
    self:installGUIHooks()
    
    debugPrint("AutoOpen Hooks - Installation complete")
end

function AutoOpenHooks:installGUIHooks()
    debugPrint("AutoOpen GUI Hooks - Installation start (Robust Version)")
    
    -- Get metatable reference for global environment
    local modEnvMeta = getmetatable(_G)
    local env = modEnvMeta.__index
    InGameMenuSettingsFrame = env.InGameMenuSettingsFrame
    
    if InGameMenuSettingsFrame == nil then
        debugPrint("WARNING: InGameMenuSettingsFrame not found!")
        return
    end
    
    -- ROBUST GUI INITIALIZATION with multiple fallback strategies
    self:installRobustGUIInitHook()
    
    -- Settings Save Hook (unchanged)
    if GameSettings and GameSettings.saveToXMLFile then
        GameSettings.saveToXMLFile = Utils.appendedFunction(
            GameSettings.saveToXMLFile, 
            self.saveSettingsToXML
        )
        debugPrint("saveToXMLFile hook installed!")
    else
        debugPrint("WARNING: GameSettings.saveToXMLFile not found!")
    end
    
    debugPrint("AutoOpen GUI Hooks - Installation complete")
end

-- New robust GUI initialization system
function AutoOpenHooks:installRobustGUIInitHook()
    debugPrint("Installing robust GUI initialization hooks...")
    
    -- Strategy 1: Hook onFrameOpen (primary method)
    if InGameMenuSettingsFrame.onFrameOpen then
        InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(
            InGameMenuSettingsFrame.onFrameOpen, 
            self.robustInitGui
        )
        debugPrint("onFrameOpen hook installed (Strategy 1)!")
    else
        debugPrint("WARNING: InGameMenuSettingsFrame.onFrameOpen not found!")
    end
    
    -- Strategy 2: Hook updateGeneralSettings as backup
    if InGameMenuSettingsFrame.updateGeneralSettings then
        InGameMenuSettingsFrame.updateGeneralSettings = Utils.appendedFunction(
            InGameMenuSettingsFrame.updateGeneralSettings, 
            self.robustUpdateGui
        )
        debugPrint("updateGeneralSettings hook installed (Strategy 2)!")
    else
        debugPrint("WARNING: InGameMenuSettingsFrame.updateGeneralSettings not found!")
    end
    
    -- Strategy 3: Timer-based fallback for mod conflicts
    self:setupTimerBasedGUIFallback()
end

-- Timer-based fallback system for mod conflicts
function AutoOpenHooks:setupTimerBasedGUIFallback()
    debugPrint("Setting up timer-based GUI fallback system...")
    
    -- Check every 2 seconds if GUI needs to be initialized
    if g_currentMission and g_currentMission.addPausedTimer then
        g_currentMission:addPausedTimer(2000, function()
            self:checkAndInitializeGUI()
        end, false)
        debugPrint("Timer-based GUI fallback system activated!")
    else
        debugPrint("WARNING: Could not setup timer-based fallback!")
    end
end

-- Check if GUI needs initialization and do it
function AutoOpenHooks:checkAndInitializeGUI()
    local success, errorMsg = pcall(function()
        -- Only proceed if mission and manager are ready
        if not g_currentMission or not g_currentMission.autoOpen then
            return
        end
        
        -- Check if settings frame exists
        local gameMenu = g_gui:getScreenByName("InGameMenuSettingsFrame")
        if not gameMenu then
            return
        end
        
        -- Check if our settings are already there
        if gameMenu[g_currentMission.autoOpen.settings.title] then
            return -- Already initialized
        end
        
        -- Check if frame is ready for initialization
        if gameMenu.generalSettingsLayout and gameMenu.generalSettingsLayout.elements then
            debugPrint("Timer fallback: Attempting GUI initialization...")
            AutoOpenSettingsManager.initGui(gameMenu, nil, g_currentMission.autoOpen)
        end
    end)
    
    if not success then
        debugPrint("ERROR in timer-based GUI check: " .. tostring(errorMsg))
    end
end

-- ============================================================================
-- ROBUST GUI Callback Functions
-- ============================================================================

-- Enhanced GUI initialization with better error handling and retry logic
function AutoOpenHooks.robustInitGui(settingsFrame, element)
    local success, errorMsg = pcall(function()
        -- Multiple checks to ensure stability
        if not g_currentMission or not g_currentMission.autoOpen then
            debugPrint("AutoOpen Manager not ready - deferring GUI init...")
            -- Setup a retry timer
            if g_currentMission and g_currentMission.addPausedTimer then
                g_currentMission:addPausedTimer(1000, function()
                    AutoOpenHooks.robustInitGui(settingsFrame, element)
                end, true) -- One-time timer
            end
            return
        end

        -- Check if AutoOpenSettingsManager is available
        if not AutoOpenSettingsManager then
            debugPrint("AutoOpenSettingsManager not available - deferring GUI init...")
            return
        end
        
        -- Check if settings frame is in a valid state
        if not settingsFrame or not settingsFrame.generalSettingsLayout then
            debugPrint("Settings frame not in valid state - deferring GUI init...")
            return
        end
        
        -- Check if already initialized to prevent double-initialization
        local manager = g_currentMission.autoOpen
        if settingsFrame[manager.settings.title] then
            debugPrint("AutoOpen GUI already initialized - skipping...")
            return
        end

        debugPrint("Robust GUI initialization starting...")
        AutoOpenSettingsManager.initGui(settingsFrame, element, manager)
        debugPrint("Robust AutoOpen GUI initialized successfully!")
    end)
    
    if not success then
        debugPrint("ERROR in robust AutoOpen initGui: " .. tostring(errorMsg))
        -- Try again later with timer fallback
        if g_currentMission and g_currentMission.addPausedTimer then
            g_currentMission:addPausedTimer(3000, function()
                AutoOpenHooks.robustInitGui(settingsFrame, element)
            end, true)
        end
    end
end

-- Enhanced GUI update with better error handling
function AutoOpenHooks.robustUpdateGui(settingsFrame)
    local success, errorMsg = pcall(function()
        -- Multiple stability checks
        if not g_currentMission or not g_currentMission.autoOpen then
            debugPrint("AutoOpen Manager not available during update - attempting initialization...")
            -- Try to initialize if not done yet
            AutoOpenHooks.robustInitGui(settingsFrame, nil)
            return
        end

        if not AutoOpenSettingsManager then
            debugPrint("AutoOpenSettingsManager not available during update")
            return
        end
        
        -- Check if GUI was initialized
        local manager = g_currentMission.autoOpen
        if not settingsFrame[manager.settings.title] then
            debugPrint("AutoOpen GUI not initialized - attempting initialization...")
            AutoOpenSettingsManager.initGui(settingsFrame, nil, manager)
            return
        end

        -- Perform the actual update
        AutoOpenSettingsManager.updateGui(settingsFrame, manager)
        debugPrint("Robust AutoOpen GUI updated successfully!")
    end)
    
    if not success then
        debugPrint("ERROR in robust AutoOpen updateGui: " .. tostring(errorMsg))
    end
end

-- Original GUI initialization (kept for compatibility)
function AutoOpenHooks.initGui(settingsFrame, element)
    -- Delegate to robust version
    AutoOpenHooks.robustInitGui(settingsFrame, element)
end

-- Original GUI update (kept for compatibility)  
function AutoOpenHooks.updateGui(settingsFrame)
    -- Delegate to robust version
    AutoOpenHooks.robustUpdateGui(settingsFrame)
end

-- Save AutoOpen settings to XML file (unchanged)
function AutoOpenHooks.saveSettingsToXML(xmlFile)
    local success, errorMsg = pcall(function()
        -- Check if AutoOpen Manager exists
        if not g_currentMission or not g_currentMission.autoOpen then
            debugPrint("AutoOpen Manager not found - skipping settings save")
            return
        end

        if g_currentMission.autoOpen.settings then
            g_currentMission.autoOpen.settings:saveToXMLFile()
            debugPrint("AutoOpen settings saved")
        end
    end)
    
    if not success then
        debugPrint("ERROR in AutoOpen saveSettingsToXML: " .. tostring(errorMsg))
    end
end

-- ============================================================================
-- Initialization & Cleanup
-- ============================================================================

function AutoOpenHooks:initialize(mission, i18n, modName, modDirectory)
    debugPrint("AutoOpen Hooks - Robust Initialization start")
    
    -- Safety check: Are all dependencies available?
    if not AutoOpenManager then
        debugPrint("ERROR: AutoOpenManager not available!")
        return
    end
    
    -- Create AutoOpen Manager and assign to mission
    mission.autoOpen = AutoOpenManager.new(mission, i18n, modName, modDirectory)
    
    -- Load Settings
    if mission.autoOpen.settings then
        mission.autoOpen.settings:loadFromXML()
        debugPrint("AutoOpen settings loaded")
    end
    
    -- Print Current Settings
    mission.autoOpen:printCurrentSettings()
    
    -- Install all hooks
    self:installHooks()
    
    debugPrint("AutoOpen Hooks - Robust Initialization complete")
end

function AutoOpenHooks:cleanup()
    debugPrint("AutoOpen Cleanup - Start")
    
    if g_currentMission and g_currentMission.autoOpen then
        g_currentMission.autoOpen:delete()
        g_currentMission.autoOpen = nil
        debugPrint("AutoOpen Manager deleted")
    end
    
    debugPrint("AutoOpen Cleanup - Complete")
end

return AutoOpenHooks