-- ============================================================================
-- AutoOpenEvent.lua - Multiplayer Event System for AutoOpen
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Multiplayer synchronization for gate operations
-- - Network event broadcasting
-- - Client/Server communication
-- - Gate open/close event types
-- ============================================================================

AutoOpenEvent = {}
local AutoOpenEvent_mt = Class(AutoOpenEvent, Event)

InitEventClass(AutoOpenEvent, "AutoOpenEvent")

-- Event Types
AutoOpenEvent.TYPE_OPEN = 1
AutoOpenEvent.TYPE_CLOSE = 2

function AutoOpenEvent.emptyNew()
    local self = Event.new(AutoOpenEvent_mt)
    return self
end

function AutoOpenEvent.new(animatedObject, eventType)
    local self = AutoOpenEvent.emptyNew()
    self.animatedObject = animatedObject
    self.eventType = eventType
    return self
end

function AutoOpenEvent:readStream(streamId, connection)
    self.animatedObject = NetworkUtil.readNodeObject(streamId)
    self.eventType = streamReadUInt8(streamId)
    self:run(connection)
end

function AutoOpenEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.animatedObject)
    streamWriteUInt8(streamId, self.eventType)
end

function AutoOpenEvent:run(connection)
    if self.animatedObject ~= nil then
        local open = self.eventType == AutoOpenEvent.TYPE_OPEN
        AutoOpen:autoOpenSet(self.animatedObject, open, true) -- true: no re-sending of event
    end
end

-- Send AutoOpen event to all clients
function AutoOpenEvent.sendEvent(animatedObject, eventType)
    local isMP = g_server ~= nil or g_client ~= nil
    if not isMP then
        return -- no MP, no network needed
    end

    if g_server ~= nil then
        -- Server: send to all clients
        g_server:broadcastEvent(AutoOpenEvent.new(animatedObject, eventType), nil, nil, animatedObject)
    else
        -- Client: send to server
        g_client:getServerConnection():sendEvent(AutoOpenEvent.new(animatedObject, eventType))
    end
end

debugPrint("AutoOpenEvent System loaded", "LOADING")